/** @file   hud.cpp
 * @brief   Implementation of Hud - class.
 * @version $Revision: 1.6 $
 * @author  Tomi Lamminsaari
 */

#include "hud.h"
#include "GfxManager.h"
#include "consts.h"
#include "GfxId.h"
#include "www_assert.h"
#include "Resource/HudSymbols.rh"
using std::vector;
using std::string;
using namespace eng2d;


namespace WeWantWar {

/** Constructor
 */
Hud::Hud() :
  m_cursorpos( 320,240 ),
  m_healthGrad( Color(80,200,50), Color(200,200,0), Color(200,0,0) ),
  m_healthLevel( 100 ),
  m_ammocount( 0 ),
  m_grenadecount( 0 ),
  m_weaponID( Weapon::W_CROWBAR ),
  m_pObjectiveMessage( 0 ),
  m_objectiveString( "" ),
  m_crosshairSprite( 0 ),
  m_comboAnimation( 0 ),
  m_bodyArmor( 0 ),
  m_maximumBodyArmor( 255 )
{
  // We create room for 10 messages by pushing 10 null-pointers to the
  // vector.
  for ( int i=0; i< 10; i++ ) {
    m_messages.push_back( 0 );
  }
  
  m_healthGrad.setBias( 75 );
}



/** Destructor
 */
Hud::~Hud()
{
  // We destroy the contents of the message-vector.
  for ( int i=0; i < m_messages.size(); i++ ) {
    delete m_messages.at(i);
    m_messages.at(i) = 0;
  }
  m_messages.clear();
  delete m_comboAnimation;
}




///
/// Public methods
/// ==============

/** Sets the position of the crosshair
 */
void Hud::crosshairPos( const Vec2D& rP )
{
  m_cursorpos = rP;
}



/** Sets the crosshair sprite.
 */
void Hud::setCrosshairSprite( RLE_SPRITE* aSprite )
{
  m_crosshairSprite = aSprite;
}


/** Sets the health level
 */
void Hud::healthLevel( int hlevel )
{
  m_healthLevel = hlevel;
}



/** Sets the active weapon
 */
void Hud::activeWeapon( Weapon::WeaponID wid )
{
  m_weaponID = wid;
}



/** Sets the ammocount
 */
void Hud::ammocount( int num )
{
  m_ammocount = num;
}



/** Sets the grenadecount
 */
void Hud::grenadecount( int num )
{
  m_grenadecount = num;
}



/** Sets the amount of bullettime left
 */
void Hud::bulletTime( int bulltime )
{
  m_bullettime = bulltime;
}



/** Adds new message
 */
void Hud::addMessage( const string& txt, int counter )
{
  for ( int i=0; i < m_messages.size(); i++ ) {
    if ( m_messages.at(i) == 0 ) {
      m_messages.at(i) = new OnScreenMessage;
      m_messages.at(i)->counter = counter;
      m_messages.at(i)->text = txt;
      return;
    }
  }
}



/** Sets the timer value
 */
void Hud::countdownValue( int v )
{
  m_countdownTimer = v;
}



/** Sets the objective text
 */
void Hud::objectiveMessage( const std::string& rTxt )
{
  if ( m_objectiveString == rTxt ) {
    // No need to update the buffer
    return;
  }
  
  m_objectiveString = rTxt;
  if ( m_pObjectiveMessage != 0 ) {
    delete m_pObjectiveMessage;
    m_pObjectiveMessage = 0;
  }
  m_pObjectiveMessage = new ObjectiveNote( 100,460, 440,480, rTxt );
}



/** Redraws the hud.
 */
void Hud::redraw( BITMAP* pB )
{
  // We draw the mouse cursor
  FONT* ammoFont = 0;
  if( GfxManager::ammoFont != 0 ) {
    ammoFont = GfxManager::ammoFont->al_FONT();
  }
  
  if ( m_crosshairSprite != 0 ) {
    int mx = static_cast<int>(m_cursorpos.x()) - ( m_crosshairSprite->w / 2 );
    int my = static_cast<int>(m_cursorpos.y()) - ( m_crosshairSprite->h / 2 );
    draw_rle_sprite( pB, m_crosshairSprite, mx, my );
  }
  
  // Draw the energybar
  int fromY = 150 - m_healthLevel;
  int toY = 150;
  for (int y=fromY; y < toY; y += 2 ) {
    hline( pB, 0,y,10, m_healthGrad.getColorAt(y, 50, 150).allC() );
  }
  
  // Draw the weapon thumbnail
  RLE_SPRITE* sprite = getWeaponSprite();
  if ( sprite != 0 ) {
    draw_rle_sprite( pB, sprite, 0, 0 );
  }
  
  // Draw grenades
  if ( m_grenadecount > 0 ) {
    sprite = GfxManager::findRleSprite( GfxId::KHudSymbols, HudSymbol_Grenade );
    if ( sprite != 0 ) {
      int gx = Display::scrWidth() - sprite->w - 2;
      draw_rle_sprite( pB, sprite, gx,0);
      textprintf_ex( pB, ammoFont, gx-20,0, makecol(230,230,230), -1, "%d", m_grenadecount );
    }
  }
  
  // Draw the ammocount
  if ( m_weaponID != Weapon::W_CROWBAR ) {
    textprintf_ex( pB, ammoFont, 150,0, makecol(230,230,230),-1, "%d", m_ammocount );
  }
  
  // We draw the onscreen messages
  int y = 450;
  for (int i=0; i < m_messages.size(); i++ ) {
    if ( m_messages.at(i) != 0 ) {
      // Draw message
      textout_ex( pB, ammoFont,
                  m_messages.at(i)->text.c_str(), 0,y, makecol(230,230,230), -1 );
      y -= 12;
    }
  }
  
  // And if countdown timer is active, we draw it.
  if ( m_countdownTimer >= 0 ) {
    int secs = m_countdownTimer / 40;
    int mins = secs / 60;
    secs = secs % 60;

    string secstr( "00" );
    string minstr( "00" );
    secstr[0] = static_cast<char>( secs / 10 ) + '0';
    secstr[1] = static_cast<char>( secs % 10 ) + '0';
    minstr[0] = static_cast<char>( mins / 10 ) + '0';
    minstr[1] = static_cast<char>( mins % 10 ) + '0';
  
    textprintf_centre_ex( pB, ammoFont,
                          320,440, makecol(230,230,230),-1,
                          "%s:%s", minstr.c_str(), secstr.c_str() );
  }
  
  // Now we draw the bullet-time indicator.
  float tickInRadians = 256.0 / static_cast<float>( Consts::MAX_BULLETTIME );
  float currentAngle = tickInRadians * static_cast<float>(m_bullettime);
  BITMAP* bulletTimer = GfxManager::findBitmap( GfxId::KHudSymbols,
                                                HudSymbol_BulletTimer );
  set_trans_blender( 0,0,0, static_cast<int>( currentAngle ) );
  draw_trans_sprite( pB, bulletTimer, 640-bulletTimer->w,480-bulletTimer->h );
  
  // Draw amount of bodyarmor
  RLE_SPRITE* armorGfx = GfxManager::findRleSprite( GfxId::KHudSymbols, HudSymbol_BodyArmor );
  if ( armorGfx != 0 && m_bodyArmor > 0 ) {
    float armorPercents = static_cast<float>( m_bodyArmor ) / static_cast<float>( m_maximumBodyArmor );
    armorPercents *= 255;
    set_trans_blender( 0,0,0, static_cast<int>(armorPercents) );
    draw_trans_rle_sprite( pB, armorGfx, (640-armorGfx->w),50 );
  }
    
  // And finally the Objective description
  if ( m_pObjectiveMessage != 0 ) {
    m_pObjectiveMessage->redraw( pB );
  }
  
  // Draw combo
  if ( m_comboAnimation != 0 ) {
    RLE_SPRITE* sprite = m_comboAnimation->currentFrame().asRleSprite();
    set_trans_blender(0,0,0, m_comboAnimation->currentFrame().iFrameAlpha);
    int x = 320 - (sprite->w / 2);
    int y = 200 - (sprite->h / 2);
    draw_trans_rle_sprite( Display::buffer, sprite, x,y );
  }
}


/** Updates the hud
 */
void Hud::update()
{
  // Update the timers of the onscreen messages and deletes the messages when
  // the counters reach zero.
  for ( int i=0; i < m_messages.size(); i++ ) {
    if ( m_messages.at(i) != 0 ) {
      m_messages.at(i)->counter -= 1;
      if ( m_messages.at(i)->counter < 0 ) {
        delete m_messages.at(i);
        m_messages.at(i) = 0;
      }
    }
  }
  
  if ( m_pObjectiveMessage != 0 ) {
    m_pObjectiveMessage->update();
  }
  
  if ( m_comboAnimation != 0 ) {
    m_comboAnimation->update();
    if ( m_comboAnimation->paused() ) {
      delete m_comboAnimation;
      m_comboAnimation = 0;
    }
  }
}

void Hud::setComboAnimation( Animation* aComboAnim )
{
  delete m_comboAnimation;
  m_comboAnimation = aComboAnim;
}

void Hud::setBodyArmor( int aBodyArmor, int aMaxBodyArmor )
{
  m_bodyArmor = aBodyArmor;
  m_maximumBodyArmor = aMaxBodyArmor;
}


///
/// Getter methods
/// ==============

/** Returns a pointer to hud sprite representing current weapon.
 */
RLE_SPRITE* Hud::getWeaponSprite() const
{
  int spriteIndex = -1;
  switch ( m_weaponID ) {
    case ( Weapon::W_CROWBAR ):
      spriteIndex = HudSymbol_Crowbar;
      break;
    case ( Weapon::W_RIFLE ):
      spriteIndex = HudSymbol_Rifle;
      break;
    case ( Weapon::W_SHOTGUN ):
      spriteIndex = HudSymbol_Shotgun; 
      break;
    case ( Weapon::W_FLAMETHROWER ):
      spriteIndex = HudSymbol_FlameThrower;
      break;
    case ( Weapon::W_MINIGUN ):
      spriteIndex = HudSymbol_Minigun;
      break;
    case ( Weapon::W_UZI ):
      spriteIndex = HudSymbol_Uzi;
      break;
    case ( Weapon::W_SNIPERRIFLE ):
      spriteIndex = HudSymbol_SniperRifle;
      break;
    case ( Weapon::W_ROCKETLAUNCHER ):
      spriteIndex = HudSymbol_RocketLauncher;
      break;
    default:
      spriteIndex = -1;
      break;
  }
  return GfxManager::findRleSprite( GfxId::KHudSymbols, spriteIndex );
}


} // end of namespace
